--[[----------------------------------------------------------------------------

ICATetherTask.lua
Tasks for Lightroom sample Tether plugin

--------------------------------------------------------------------------------

ADOBE SYSTEMS INCORPORATED
 Copyright 2019 Adobe Incorporated
 All Rights Reserved.

NOTICE: Adobe permits you to use, modify, and distribute this file in accordance
with the terms of the Adobe license agreement accompanying it. If you have received
this file from a source other than Adobe, then your use, modification, or distribution
of it requires the prior written permission of Adobe.

------------------------------------------------------------------------------]]

local LrLogger = import 'LrLogger'
local LrPathUtils = import 'LrPathUtils'
local LrFileUtils = import "LrFileUtils"

local LrRemoteCommunication = import 'LrRemoteCommunication'

local LrTableUtils = import "LrTableUtils"

--============================================================================--

local ICATetherTask = {}

-------------------------------------------------------------------------------

local logger = LrLogger 'leica-tether'
logger:enable ("logfile")

local launchAndConnectToTetherServer

-------------------------------------------------------------------------------

function ICATetherTask.init( propertyTable, pluginPath, completionCallback )
	-- Attempt to find and launch the tether application

	logger:trace( "plugin path: ", pluginPath, " callback: ", completionCallback )

	ICATetherTask._connectedToPluginApp = false
	ICATetherTask._sentListDevicesMessage = false
	ICATetherTask._propertyTable = propertyTable

	launchAndConnectToTetherServer( pluginPath, completionCallback )
end

-------------------------------------------------------------------------------

function launchAndConnectToTetherServer( pluginPath, completionCallback )
	-- Attempt a connection

	local taskPath
	if MAC_ENV then

		-- if DEBUG then
		-- 	local testPath
		-- 	testPath = LrPathUtils.child( pluginPath, "../../../../../tether_ica/Debug/tether-ica.app/Contents/MacOS/tether-ica" )
		-- 	logger:trace( "debug build - testing: ", testPath )
		-- 	if LrFileUtils.exists( testPath ) == "file" then
		-- 		taskPath = testPath
		-- 	end
		-- end

		if not taskPath then
			taskPath = LrPathUtils.child( pluginPath, "leica-tether.app/Contents/MacOS/leica-tether" )
		end

		logger:trace( "launching plugin app with path: ", taskPath )
	else
		taskPath = pluginPath .. "/leica-tether.exe"

		logger:trace( "launching plugin app with path: ", taskPath )
	end

	local function failureCallback()
		-- The task crashed. Let's try relaunching it
		launchAndConnectToTetherServer( pluginPath, completionCallback )
	end

	LrRemoteCommunication.spawnTaskAndConnect( "leica-tether", taskPath,
		function( serverRef )
			logger:trace( "launchAndConnectToTetherServer - serverRef: ", serverRef )
			ICATetherTask._serverConnection = serverRef
			if completionCallback then completionCallback( serverRef ~= nil ) end
		end,
		failureCallback )

	-- Uncomment this to halt after starting up the task, giving you a chance to attach the
	-- debugger.
	-- halt()

end

-------------------------------------------------------------------------------

function ICATetherTask.shutdown( propertyTable )
	-- Send a message to the server asking it to shut down
	if ICATetherTask._serverConnection then
		logger:trace( "ICATetherTask.shutdown" )
		LrRemoteCommunication.closeNamedConnection( ICATetherTask._serverConnection, "leica-tether" )
	end
end

-------------------------------------------------------------------------------

function ICATetherTask.setCallback( propertyTable, callback )
	ICATetherTask._hostCallback = callback
end

-------------------------------------------------------------------------------

function ICATetherTask.tetherCallback( propertyTable, messageType, result )
	logger:trace( "ICATetherTask.tetherCallback - message: ", messageType )

	if messageType == "objectWasAdded" then
		if ICATetherTask._hostCallback then
			local device = ICATetherTask._connectedDevice
			if not device then
				logger:error( "ICATetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			if not device.captures then
				device.captures = {}
			end
			if not device.captures[ captureId ] then
				device.captures[ captureId ] = {}
			end

			logger:tracef( "objectWasAdded - device %s, device.captures = %s, device.captures[%s] = %s", device, device.captures, captureId, device.captures[captureId] )

			ICATetherTask._hostCallback( "objectWasAdded", result )
		end
	elseif messageType == "fileDownloadComplete" then
		if ICATetherTask._hostCallback then
			logger:trace( "ICATetherTask.tetherCallback - fileDownloadComplete: ", result.file_path )
			local device = ICATetherTask._connectedDevice

			if not device then
				logger:error( "ICATetherTask.tetherCallback: Could not locate device." )
				return
			end
			local captureId = result.capture_id
			local capture = device.captures[ captureId ]

			if not capture then
				logger:error( "ICATetherTask.tetherCallback: could not find capture id: ", captureId )
				return
			end

			ICATetherTask._hostCallback( "fileDownloadComplete",
							{ captureId = captureId,
								filePath = result.file_path,
								fileType = LrPathUtils.extension( result.file_path ),
								fileDownloadTime = result.file_download_time,
								status = "done" } )
		end
	elseif messageType == "deviceAdded" then
		logger:trace( "ICATetherTask.tetherCallback - deviceAdded: ", result.device_id )
		if ICATetherTask._propertyTable.availableDevices then
			local forceUpdate = false
			for i, device in ipairs( ICATetherTask._propertyTable.availableDevices ) do
				if device.device_id == result.device_id then
					forceUpdate = true
				end
			end
			if not forceUpdate then
				table.insert( ICATetherTask._propertyTable.availableDevices, result )
			end
			LrTableUtils.debugDumpTable( ICATetherTask._propertyTable.availableDevices )
			logger:trace( "- deviceAdded forceUpdate: ", forceUpdate )
			ICATetherTask._hostCallback( "availableDevicesChanged", forceUpdate )

		end
	elseif messageType == "deviceRemoved" then
		logger:trace( "ICATetherTask.tetherCallback - deviceRemoved: ", result.device_id )
		local deviceId = result.device_id
		if deviceId and ICATetherTask._propertyTable.availableDevices then
			for i, v in ipairs( ICATetherTask._propertyTable.availableDevices ) do
				if v.device_id == deviceId then
					table.remove( ICATetherTask._propertyTable.availableDevices, i )
					ICATetherTask._hostCallback( "availableDevicesChanged" )
					break
				end
			end
		end
	elseif messageType == "cameraSettingsChanged" then
		local settings = result
		ICATetherTask._hostCallback( "cameraSettingsChanged", settings )
	else
		ICATetherTask._hostCallback( messageType, result )
	end
end

-------------------------------------------------------------------------------


function ICATetherTask.pollForMessages( propertyTable )
--	logger:trace( 'ICATetherTask.pollForMessages()' )
	pcall( function()
		local message, params = LrRemoteCommunication.pollForMessage( ICATetherTask._serverConnection )

		if message then
			logger:trace( 'ICATetherTask.pollForMessages() - received message: ', message )

			ICATetherTask.tetherCallback( propertyTable, message, params )
		end
	end )
end

-------------------------------------------------------------------------------

function ICATetherTask.queryDevices( propertyTable )
	logger:trace( 'ICATetherTask.queryDevices' )

	-- If we're still not connected, wait until later
	if not ICATetherTask._connectedToPluginApp then
		if ICATetherTask._serverConnection then
			ICATetherTask._connectedToPluginApp = true
		else
			return
		end
	end

	logger:trace( "ICATetherTask.queryDevices : 1" )

	local function listDevicesCallback( messageType, result )

		logger:trace( "listDevicesCallback called with result: ", messageType )

		if messageType == "ok" then
			ICATetherTask._propertyTable.availableDevices = result and result.devices or {}
			ICATetherTask._hostCallback( "availableDevicesChanged" )

			LrTableUtils.debugDumpTable( ICATetherTask._propertyTable.availableDevices, "Device Table" )
		end
	end

	if not ICATetherTask._sentListDevicesMessage then
		logger:trace( 'ICATetherTask.queryDevices: sending message to server with callback:', listDevicesCallback )
		LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "listDevices", {}, listDevicesCallback )
		ICATetherTask._sentListDevicesMessage = true
	end

	if ICATetherTask._propertyTable.availableDevices then
		local devices = {}

		for i, v in ipairs( ICATetherTask._propertyTable.availableDevices ) do
			table.insert( devices, { id = v.device_id, name = v.name } )
		end

		return { status = 'ok', devices = devices }
	else
		return { status = 'notReady' }
	end
end

-------------------------------------------------------------------------------

function ICATetherTask.connectDevice( propertyTable, deviceId, callback )
	logger:trace( 'ICATetherTask.connectDevice' )

	local function openDeviceCallback( messageType, result )
		if messageType == "ok" then
			ICATetherTask._sessionId = result and result._sessionId
			if callback and result and result.camera_name then
				ICATetherTask._connectedDevice.name = result.camera_name
				callback( result.camera_name )
			end
		end
	end

	for i,v in ipairs( ICATetherTask._propertyTable.availableDevices ) do
		if v.device_id == deviceId then
			ICATetherTask._connectedDevice = v
		end
	end

	LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "connectDevice", { device_id = deviceId }, openDeviceCallback )

	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function ICATetherTask.disconnectDevice( propertyTable, deviceId )
	logger:trace( 'ICATetherTask.disconnectDevice' )

	LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "disconnectDevice", { device_id = deviceId }, nil )
end

-------------------------------------------------------------------------------

function ICATetherTask.doCapture( propertyTable, deviceId )
	logger:trace( 'ICATetherTask.doCapture' )

	local function doCaptureCallback( messageType, result )
		if messageType == "ok" then
		end
	end

	LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "doCapture", { device_id = deviceId }, doCaptureCallback )

	return { status = 'ok' }
end

-------------------------------------------------------------------------------

function ICATetherTask.startDownload( propertyTable, deviceId, captureId )
	logger:trace( 'ICATetherTask.startDownload' )

	local device = ICATetherTask._connectedDevice
	local capture = device and device.captures[ captureId ]
	if capture then
		LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "startDownload", { capture_id = captureId },
					function( messageType, result ) if messageType == "error" then capture.errorCode = result end end )
	end
end

-------------------------------------------------------------------------------

function ICATetherTask.setCameraSettings( propertyTable, deviceId, condition, valueIndex )
	logger:trace( 'ICATetherTask.setCameraSettings' )

	local function happyCallback( messageType, result )
		logger:trace( 'ICATetherTask.setCameraSettings: received result - ', messageType, result )
		if messageType == "ok" then
		end
	end

	local params =
		{
			device_id = deviceId,
			param_key = condition,
			param_value = valueIndex
		}

	LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "updateCameraSetting", params, happyCallback )
end

--------------------------------------------------------------------------------

function ICATetherTask.getCameraSettings( propertyTable, deviceId, callbackFn )
	logger:trace( 'ICATetherTask.getCameraSettings' )

	LrRemoteCommunication.sendMessageToServer( ICATetherTask._serverConnection, "getCameraSettings", { device_id = deviceId },
			function( messageType, result )
				logger:trace( 'ICATetherTask.getCameraSettings: received result - ', messageType, result )

				if messageType == "error" then
					callbackFn( nil, result.error_code )
				else
					callbackFn( result )
				end
			end )
end

-------------------------------------------------------------------------------

return ICATetherTask
