// SPDX-License-Identifier: GPL-2.0
//
// Copyright (C) 2019 Linaro Ltd.
// Copyright (C) 2019 Socionext Inc.
// Copyright (c) 2022 Panasonic Corporation

#include <linux/bits.h>
#include <linux/clk.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/iopoll.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/of_dma.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/types.h>
#include <linux/bitfield.h>

#include "virt-dma.h"

/* DMA_MEMCPY_SG			*/
#include <linux/arm-smccc.h>
#include <asm/io.h>
#define SMC_CMD_PANA_SECURE_WRITE64     (0x8200C001)

/************************************************************************
 *  DEBUG Define macro
 ***********************************************************************/
//#define DEBUG_TRAN_SPEED
//#define DEBUG_MODE

#if defined( DEBUG_MODE )
#define DBG_PRINT( fmt, ... )  printk( fmt, ##__VA_ARGS__ )
#else
#define DBG_PRINT( fmt, ... )
#endif	/* DEBUG_MODE	*/

/************************************************************************
 *  Define macro
 ***********************************************************************/

#define MLB_HDMAC_DMACR		0x0	/* global */
#define MLB_HDMAC_DE		BIT(31)
#define MLB_HDMAC_DS		BIT(30)
#define MLB_HDMAC_PR		BIT(28)
#define MLB_HDMAC_DH		GENMASK(27, 24)

#define MLB_HDMAC_CH_STRIDE	0x10

#define MLB_HDMAC_DMACA		0x0	/* channel */
#define MLB_HDMAC_EB		BIT(31)
#define MLB_HDMAC_PB		BIT(30)
#define MLB_HDMAC_ST_M10V	BIT(29)
#define MLB_HDMAC_IS_M10V	GENMASK(28, 24)
#define MLB_HDMAC_IS_M20V	GENMASK(29, 24)
#define MLB_HDMAC_BT		GENMASK(23, 20)
#define MLB_HDMAC_BC		GENMASK(19, 16)
#define MLB_HDMAC_TC		GENMASK(15, 0)
#define MLB_HDMAC_DMACB		0x4
#define MLB_HDMAC_TT		GENMASK(31, 30)
#define MLB_HDMAC_MS		GENMASK(29, 28)
#define MLB_HDMAC_TW		GENMASK(27, 26)
#define MLB_HDMAC_FS		BIT(25)
#define MLB_HDMAC_FD		BIT(24)
#define MLB_HDMAC_RC		BIT(23)
#define MLB_HDMAC_RS		BIT(22)
#define MLB_HDMAC_RD		BIT(21)
#define MLB_HDMAC_EI		BIT(20)
#define MLB_HDMAC_CI		BIT(19)
#define HDMAC_PAUSE		0x7
#define MLB_HDMAC_SS		GENMASK(18, 16)
#define MLB_HDMAC_SP		GENMASK(15, 12)
#define MLB_HDMAC_DP		GENMASK(11, 8)
#define MLB_HDMAC_ST_M20V	BIT(7)
#define MLB_HDMAC_DMACSA	0x8
#define MLB_HDMAC_DMACDA	0xc

#define MLB_HDMAC_BUSWIDTHS		(BIT(DMA_SLAVE_BUSWIDTH_1_BYTE) | \
					BIT(DMA_SLAVE_BUSWIDTH_2_BYTES) | \
					BIT(DMA_SLAVE_BUSWIDTH_4_BYTES))

#define MLB_M10V_CODE		10
#define MLB_M20V_CODE		20

#define MLB_M20V_HDMAC_CH_BASE		0x100
#define MLB_M20V_HDMAC_CH_STRIDE	0x20

struct product_code {
	unsigned char number;
};

static struct product_code product_code_m10v = {
	.number = MLB_M10V_CODE
};

static struct product_code product_code_m20v = {
	.number = MLB_M20V_CODE
};

struct milbeaut_hdmac_desc {
	struct virt_dma_desc vd;
	struct scatterlist *sgl;
	unsigned int sg_len;
	unsigned int sg_cur;
	enum dma_transfer_direction dir;
	/* DMA_MEMCPY_SG	*/
	struct scatterlist	*sgl_dst;
	int					data_width;	/*  1:Byte 2:Half-word 4:Word		*/
#ifdef DEBUG_TRAN_SPEED
	size_t				len;
	ktime_t				s_ktime;
#endif	/* DEBUG_TRAN_SPEED	*/
};

struct milbeaut_hdmac_chan {
	struct virt_dma_chan vc;
	struct milbeaut_hdmac_device *mdev;
	struct milbeaut_hdmac_desc *md;
	void __iomem *reg_ch_base;
	unsigned int slave_id;
	struct dma_slave_config	cfg;
};

struct milbeaut_hdmac_device {
	struct dma_device ddev;
	struct clk *clk;
	const struct product_code *soc_data;
	void __iomem *reg_base;
	struct milbeaut_hdmac_chan channels[0]; };

#ifdef DEBUG_TRAN_SPEED
static inline uint64_t hdmac_BYTEs(s64 runtime, uint64_t len);
/*-----------------------------------------------------------------------------
 * DEBUG SPEED CHECK BASE FUNCTION
 -----------------------------------------------------------------------------*/
static inline uint64_t hdmac_persec(s64 runtime, unsigned int val)
{
    uint64_t    per_sec = 1000000;

    if (runtime <= 0) {
        return 0;
    }
    /* drop precision until runtime is 32-bits */
    while (runtime > UINT_MAX) {
        runtime >>= 1;
        per_sec <<= 1;
    }
    per_sec *= val;
    do_div(per_sec, runtime);
    return per_sec;
}

static uint64_t hdmac_BYTEs(s64 runtime, uint64_t len)
{
    return hdmac_persec(runtime, len);
}
#endif	/* DEBUG_TRAN_SPEED	*/

static inline void dch_writel_relaxed(u32 value,
		volatile void __iomem *addr, unsigned int reg,
		struct milbeaut_hdmac_device *mdev)
{
	/* Absorb the difference between M10V and M20V */
	if (mdev->soc_data->number == MLB_M20V_CODE) {
		switch (reg) {
		case MLB_HDMAC_DMACSA:
			addr += 0x8;
			break;
		case MLB_HDMAC_DMACDA:
			addr += 0xC;
			break;
		}
	}
	writel_relaxed(value, addr);
}

static inline u32 dch_readl_relaxed(const volatile void __iomem *addr,
	unsigned int reg, struct milbeaut_hdmac_device *mdev)
{
	/* Absorb the difference between M10V and M20V */
	if (mdev->soc_data->number == MLB_M20V_CODE) {
		switch (reg) {
		case MLB_HDMAC_DMACSA:
			addr += 0x8;
			break;
		case MLB_HDMAC_DMACDA:
			addr += 0xC;
			break;
		}
	}
	return readl_relaxed(addr);
}

static struct milbeaut_hdmac_chan *
to_milbeaut_hdmac_chan(struct virt_dma_chan *vc) {
	return container_of(vc, struct milbeaut_hdmac_chan, vc); }

static struct milbeaut_hdmac_desc *
to_milbeaut_hdmac_desc(struct virt_dma_desc *vd) {
	return container_of(vd, struct milbeaut_hdmac_desc, vd); }

/* mc->vc.lock must be held by caller */ static struct
milbeaut_hdmac_desc * milbeaut_hdmac_next_desc(struct
milbeaut_hdmac_chan *mc) {
	struct virt_dma_desc *vd;

	vd = vchan_next_desc(&mc->vc);
	if (!vd) {
		mc->md = NULL;
		return NULL;
	}

	list_del(&vd->node);

	mc->md = to_milbeaut_hdmac_desc(vd);

	return mc->md;
}
/* noc address set  */
static inline int milbeaut_noc_addr_set(dma_addr_t reg_addr, unsigned long noc_addr)
{
    struct arm_smccc_res    res;
    arm_smccc_smc(SMC_CMD_PANA_SECURE_WRITE64,
                (unsigned long)reg_addr,        // update reggster addr
                (uint64_t)noc_addr,             // update value
                0,                              // $BBh;M0z?t$O;HMQIT2D(B
                0xfffffffffull,
                0, 0, 0, &res);
    return res.a0;
}

/* mc->vc.lock must be held by caller */
static void milbeaut_chan_start_memcpy_sg(struct milbeaut_hdmac_chan *mc,
				struct milbeaut_hdmac_desc *md)
{
	struct scatterlist *sg_src, *sg_dst;
	u32 cb, ca, src_addr, dest_addr, len;
	u32 width, burst;
	u32	tmp_tc = 0;
	u32	tmp_bc = 0;
	u32	tmp_bt = 0;
	u32	mode_select = 0;
	u32	input_select = 0;
	u32	tmp_block_size = 0;

	sg_src = &md->sgl[md->sg_cur];
	sg_dst = &md->sgl_dst[md->sg_cur];

	/* Noc address Range  */
	if (sg_src->dma_address) {
		milbeaut_noc_addr_set(sg_src->dma_address, sg_src->page_link);
	}
	if (sg_dst->dma_address) {
		milbeaut_noc_addr_set(sg_dst->dma_address, sg_dst->page_link);
	}
	DBG_PRINT("DBG[%s] SG%d Noc SRC addr:%08llx val:0x%lx xdma1:0x%x\n", __FUNCTION__,
				md->sg_cur, sg_src->dma_address, sg_src->page_link, sg_src->offset);
	DBG_PRINT("DBG[%s] SG%d Noc DST addr:%08llx val:0x%lx xdma1:0x%x\n", __FUNCTION__,
				md->sg_cur, sg_dst->dma_address, sg_dst->page_link, sg_dst->offset);

	/* DMACB MS:Mode select 			*/
	/*  0:Block transfer				*/
	/* *1:Burst transfer				*/
	/*  2:Demand transfer				*/
	mode_select = 1;

	/* Transfer Width					*/
	/*  1:Byte 2:Half-word 4:Word		*/
	width	 = md->data_width;

	/* DMACA BT:Beat Type 				*/
	/* *16: --> 0xf :INCR16				*/
	/*   8: --> 0xd :INCR8				*/
	/*   4: --> 0xb :INCR4				*/
	burst	 = 16;

	/* DMACA IS:Input Select			*/
	/*  0:Software						*/
	input_select =  0;

	/*----------------------------------*/
	/* src/dst_address					*/
	/* transfer size					*/
	/*----------------------------------*/
	src_addr = sg_src->offset;
	dest_addr = sg_dst->offset;
	len = sg_src->length;

#ifdef DEBUG_TRAN_SPEED
	md->len += len;
#endif	/* DEBUG_TRAN_SPEED	*/
	/************************************/
	/* DMACA Preparation				*/
	if (mode_select != 2) {
		if (burst == 16)
			tmp_bt = 0x0f;
		else if (burst == 8)
			tmp_bt = 0x0d;
		else if (burst == 4)
			tmp_bt = 0x0b;
		else
			tmp_bt = 0x0f;
	}
	tmp_block_size = burst * width;
	if ((mode_select != 2) && (len > tmp_block_size)) {
		/* 0: Block transfer	*/
		/* 1: Burst transfer	*/
		tmp_bc = burst - 1;
		tmp_tc = (len / tmp_block_size) - 1;
	}
	else {
		tmp_bt = 0;
		tmp_bc = 0;
		if (width == 4) {
			if ((len % width) != 0) {
				width = 2;
				if ((len % width) != 0) {
					width = 1;
				}
			}
		}
		else if (width == 2) {
			if ((len % width) != 0) {
				width = 1;
			}
		}
		tmp_tc = (len / width) - 1;
	}
	DBG_PRINT("DBG[%s] len(0x%x) width(%d) tmp_bc(0x%x) tmp_tc(0x%x) tmp_block_size(0x%x)\n",
			__FUNCTION__, len, width, tmp_bc, tmp_tc, tmp_block_size);

	/*----------------------------------*/
	/* DMACR Write						*/
	/*   DE:DMA Enable					*/
	/*----------------------------------*/
	writel_relaxed(MLB_HDMAC_DE, mc->mdev->reg_base + MLB_HDMAC_DMACR);

	/*----------------------------------*/
	/* DMACSA Write						*/
	/*  DMAC Source Address				*/
	/*----------------------------------*/
	/* DMACSA DMACSA:DMAC Source Address		*/
	dch_writel_relaxed(src_addr, mc->reg_ch_base + MLB_HDMAC_DMACSA,
			MLB_HDMAC_DMACSA, mc->mdev);

	/*----------------------------------*/
	/* DMACDA Write						*/
	/*  DMAC Destination Address		*/
	/*----------------------------------*/
	dch_writel_relaxed(dest_addr, mc->reg_ch_base + MLB_HDMAC_DMACDA,
			MLB_HDMAC_DMACDA, mc->mdev);

	/************************************/
	/* DMACB Preparation				*/
	/* DMACB MS:Mode Select				*/
	cb = FIELD_PREP(MLB_HDMAC_MS, mode_select);

	/* DMACB TW:Transfer Width			*/
	cb |= FIELD_PREP(MLB_HDMAC_TW, (width >> 1));

	/* DMACB							*/
	/* EI:Error Interrupt	   1:enable	*/
	/* CI:Completion Interrupt 1:enable	*/
	cb |= MLB_HDMAC_EI | MLB_HDMAC_CI;

	/* DMACB ST: Software Trigger		*/
	/*  		 1:Software request		*/
	cb |= MLB_HDMAC_ST_M20V;
	/*----------------------------------*/
	/* DMACB Write						*/
	/*----------------------------------*/
	dch_writel_relaxed(cb, mc->reg_ch_base + MLB_HDMAC_DMACB,
			MLB_HDMAC_DMACB, mc->mdev);


	/* DMACA IS:Input Select	*/
	if (mc->mdev->soc_data->number == MLB_M20V_CODE)
		ca = FIELD_PREP(MLB_HDMAC_IS_M20V, input_select);
	else /* MLB_M10V_CODE */
		ca = FIELD_PREP(MLB_HDMAC_IS_M10V, input_select);

	/* DMACA BT:Beat Type		*/
	if (tmp_bt) {
		ca |= FIELD_PREP(MLB_HDMAC_BT, tmp_bt);
	}
	/* DMACA BC:Block Count		*/
	ca |= FIELD_PREP(MLB_HDMAC_BT, tmp_bc);

	/* DMACA TC:Transfer Count	*/
	ca |= FIELD_PREP(MLB_HDMAC_TC, tmp_tc);

	/*----------------------------------*/
	/* DMACA write						*/
	/*----------------------------------*/
	writel_relaxed(ca, mc->reg_ch_base + MLB_HDMAC_DMACA);

	DBG_PRINT("DBG[%s] MLB_HDMAC_DMACA :reg_ch_base(0x%lx)=0x%x MLB_HDMAC_DMACA(0x%x)\n",
			__FUNCTION__, mc->reg_ch_base, ca, MLB_HDMAC_DMACA);
	DBG_PRINT("DBG[%s] mc->mdev->soc_data->number(%d) MLB_M20V_CODE(%d)\n",
			__FUNCTION__, mc->mdev->soc_data->number, MLB_M20V_CODE);

	/************************************/
	/* DMACA Preparation				*/
	/* DMACA EB:Enable Bit				*/
	ca |= MLB_HDMAC_EB;

#ifdef DEBUG_TRAN_SPEED
	if (md->sg_cur == 0) {
		md->s_ktime = ktime_get();
	}
#endif	/* DEBUG_TRAN_SPEED	*/
	/*----------------------------------*/
	/* DMACA write						*/
	/*----------------------------------*/
	writel_relaxed(ca, mc->reg_ch_base + MLB_HDMAC_DMACA);
}

/* mc->vc.lock must be held by caller */ static void
milbeaut_chan_start(struct milbeaut_hdmac_chan *mc,
				struct milbeaut_hdmac_desc *md)
{
	struct scatterlist *sg;
	u32 cb, ca, src_addr, dest_addr, len;
	u32 width, burst;

	if (md->dir == DMA_MEM_TO_MEM) {
		milbeaut_chan_start_memcpy_sg(mc, md);
		return;
	}
	sg = &md->sgl[md->sg_cur];
	len = sg_dma_len(sg);

	cb = MLB_HDMAC_CI | MLB_HDMAC_EI;
	if (md->dir == DMA_MEM_TO_DEV) {
		cb |= MLB_HDMAC_FD;
		width = mc->cfg.dst_addr_width;
		burst = mc->cfg.dst_maxburst;
		src_addr = sg_dma_address(sg);
		dest_addr = mc->cfg.dst_addr;
	} else {
		cb |= MLB_HDMAC_FS;
		width = mc->cfg.src_addr_width;
		burst = mc->cfg.src_maxburst;
		src_addr = mc->cfg.src_addr;
		dest_addr = sg_dma_address(sg);
	}
	cb |= FIELD_PREP(MLB_HDMAC_TW, (width >> 1));
	cb |= FIELD_PREP(MLB_HDMAC_MS, 2);

	writel_relaxed(MLB_HDMAC_DE, mc->mdev->reg_base + MLB_HDMAC_DMACR);
	dch_writel_relaxed(src_addr, mc->reg_ch_base + MLB_HDMAC_DMACSA,
			MLB_HDMAC_DMACSA, mc->mdev);
	dch_writel_relaxed(dest_addr, mc->reg_ch_base + MLB_HDMAC_DMACDA,
			MLB_HDMAC_DMACDA, mc->mdev);
	dch_writel_relaxed(cb, mc->reg_ch_base + MLB_HDMAC_DMACB,
			MLB_HDMAC_DMACB, mc->mdev);

	if (mc->mdev->soc_data->number == MLB_M20V_CODE)
		ca = FIELD_PREP(MLB_HDMAC_IS_M20V, mc->slave_id);
	else /* MLB_M10V_CODE */
		ca = FIELD_PREP(MLB_HDMAC_IS_M10V, mc->slave_id);
	if (burst == 16)
		ca |= FIELD_PREP(MLB_HDMAC_BT, 0xf);
	else if (burst == 8)
		ca |= FIELD_PREP(MLB_HDMAC_BT, 0xd);
	else if (burst == 4)
		ca |= FIELD_PREP(MLB_HDMAC_BT, 0xb);
	burst *= width;
	ca |= FIELD_PREP(MLB_HDMAC_TC, (len / burst - 1));
	dch_writel_relaxed(ca, mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	ca |= MLB_HDMAC_EB;
	dch_writel_relaxed(ca, mc->reg_ch_base + MLB_HDMAC_DMACA,
		MLB_HDMAC_DMACA, mc->mdev); }

/* mc->vc.lock must be held by caller */ static void 
milbeaut_hdmac_start(struct milbeaut_hdmac_chan *mc) {
	struct milbeaut_hdmac_desc *md;

	md = milbeaut_hdmac_next_desc(mc);
	if (md)
		milbeaut_chan_start(mc, md);
}

static irqreturn_t milbeaut_hdmac_interrupt(int irq, void *dev_id) {
	struct milbeaut_hdmac_chan *mc = dev_id;
	struct milbeaut_hdmac_desc *md;
	u32 val;

	spin_lock(&mc->vc.lock);
#ifdef DEBUG_TRAN_SPEED
	md = mc->md;
	if (!md) {
		goto out;
	}
	if (md->dir == DMA_MEM_TO_MEM) {
		if ((md->sg_cur + 1) >= md->sg_len) {
            s64         s_runtime;
            uint64_t    s_tran_speed;
            md->s_ktime	= ktime_sub(ktime_get(), md->s_ktime);
            s_runtime   = ktime_to_us(md->s_ktime);
            s_tran_speed = hdmac_BYTEs(s_runtime, md->len);

            printk("[DBG %s] DMA transfer speed %llu Byte/s %llu KB/s :info runtime(%llu micro-sec) tran-size(0x%lx)\n",
                __FUNCTION__, s_tran_speed, (s_tran_speed>>10), s_runtime, md->len);
		}
	}
#endif	/* DEBUG_TRAN_SPEED	*/

	/* Ack and Disable irqs */
	val = dch_readl_relaxed(mc->reg_ch_base + MLB_HDMAC_DMACB,
			MLB_HDMAC_DMACB, mc->mdev);
	val &= ~(FIELD_PREP(MLB_HDMAC_SS, HDMAC_PAUSE));
	dch_writel_relaxed(val, mc->reg_ch_base + MLB_HDMAC_DMACB,
			MLB_HDMAC_DMACB, mc->mdev);
	val &= ~MLB_HDMAC_EI;
	val &= ~MLB_HDMAC_CI;
	dch_writel_relaxed(val, mc->reg_ch_base + MLB_HDMAC_DMACB,
			MLB_HDMAC_DMACB, mc->mdev);

	md = mc->md;
	if (!md)
		goto out;

	md->sg_cur++;

	if (md->sg_cur >= md->sg_len) {
		vchan_cookie_complete(&md->vd);
		md = milbeaut_hdmac_next_desc(mc);
		if (!md)
			goto out;
	}

	milbeaut_chan_start(mc, md);

out:
	spin_unlock(&mc->vc.lock);
	return IRQ_HANDLED;
}

static void milbeaut_hdmac_free_chan_resources(struct dma_chan *chan) {
	vchan_free_chan_resources(to_virt_chan(chan));
}

static int
milbeaut_hdmac_chan_config(struct dma_chan *chan, struct
dma_slave_config *cfg) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_chan *mc = to_milbeaut_hdmac_chan(vc);

	spin_lock(&mc->vc.lock);
	mc->cfg = *cfg;
	spin_unlock(&mc->vc.lock);

	return 0;
}

static int milbeaut_hdmac_chan_pause(struct dma_chan *chan) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_chan *mc = to_milbeaut_hdmac_chan(vc);
	u32 val;

	spin_lock(&mc->vc.lock);
	val = dch_readl_relaxed(mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	val |= MLB_HDMAC_PB;
	dch_writel_relaxed(val, mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	spin_unlock(&mc->vc.lock);

	return 0;
}

static int milbeaut_hdmac_chan_resume(struct dma_chan *chan) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_chan *mc = to_milbeaut_hdmac_chan(vc);
	u32 val;

	spin_lock(&mc->vc.lock);
	val = dch_readl_relaxed(mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	val &= ~MLB_HDMAC_PB;
	dch_writel_relaxed(val, mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	spin_unlock(&mc->vc.lock);

	return 0;
}

static struct dma_async_tx_descriptor *
milbeaut_hdmac_prep_slave_sg(struct dma_chan *chan, struct scatterlist *sgl,
			     unsigned int sg_len,
			     enum dma_transfer_direction direction,
			     unsigned long flags, void *context) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_desc *md;
	int i;

	if (!is_slave_direction(direction))
		return NULL;

	md = kzalloc(sizeof(*md), GFP_NOWAIT);
	if (!md)
		return NULL;

	md->sgl = kzalloc(sizeof(*sgl) * sg_len, GFP_NOWAIT);
	if (!md->sgl) {
		kfree(md);
		return NULL;
	}

	for (i = 0; i < sg_len; i++)
		md->sgl[i] = sgl[i];

	md->sg_len = sg_len;
	md->dir = direction;

	return vchan_tx_prep(vc, &md->vd, flags); }

static struct dma_async_tx_descriptor *
milbeaut_hdmac_prep_memcpy_sg(struct dma_chan *chan, struct scatterlist *sgl_src,
				 struct scatterlist *sgl_dst,
			     unsigned int sg_len,
			     enum dma_transfer_direction direction,
			     unsigned long flags, int data_width) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_desc *md;
	int i;
	size_t				sgl_size;
	struct scatterlist	*to_sgl_src;
	struct scatterlist	*to_sgl_dst;

	DBG_PRINT("DBG[%s] START sg_len(%d)\n", __FUNCTION__, sg_len);
		
	md = kzalloc(sizeof(*md), GFP_NOWAIT);
	if (!md)
		return NULL;

	sgl_size = sizeof(struct scatterlist) * sg_len;
	to_sgl_src = md->sgl = kzalloc(sgl_size, GFP_NOWAIT);
	if (!md->sgl) {
		kfree(md);
		return NULL;
	}
	to_sgl_dst = md->sgl_dst = kzalloc(sgl_size, GFP_NOWAIT);
	if (!md->sgl_dst) {
		kfree(md->sgl_dst);
		kfree(md);
		return NULL;
	}

	for (i = 0; i < sg_len; i++){
		*to_sgl_src++ = *sgl_src++;
		*to_sgl_dst++ = *sgl_dst++;
	}

	md->sg_len = sg_len;
	md->dir = direction;
	md->data_width = data_width;

	return vchan_tx_prep(vc, &md->vd, flags); }

static int milbeaut_hdmac_terminate_all(struct dma_chan *chan) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_chan *mc = to_milbeaut_hdmac_chan(vc);
	unsigned long flags;
	u32 val;

	LIST_HEAD(head);

	spin_lock_irqsave(&vc->lock, flags);

	val = dch_readl_relaxed(mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);
	val &= ~MLB_HDMAC_EB; /* disable the channel */
	dch_writel_relaxed(val, mc->reg_ch_base + MLB_HDMAC_DMACA,
			MLB_HDMAC_DMACA, mc->mdev);

	if (mc->md) {
		vchan_terminate_vdesc(&mc->md->vd);
		mc->md = NULL;
	}

	vchan_get_all_descriptors(vc, &head);

	spin_unlock_irqrestore(&vc->lock, flags);

	vchan_dma_desc_free_list(vc, &head);

	return 0;
}

static void milbeaut_hdmac_synchronize(struct dma_chan *chan) {
	vchan_synchronize(to_virt_chan(chan));
}

static enum dma_status milbeaut_hdmac_tx_status(struct dma_chan *chan,
						dma_cookie_t cookie,
						struct dma_tx_state *txstate)
{
	struct virt_dma_chan *vc;
	struct virt_dma_desc *vd;
	struct milbeaut_hdmac_chan *mc;
	struct milbeaut_hdmac_desc *md = NULL;
	enum dma_status stat;
	unsigned long flags;
	int i;

	stat = dma_cookie_status(chan, cookie, txstate);
	/* Return immediately if we do not need to compute the residue. */
	if (stat == DMA_COMPLETE || !txstate)
		return stat;

	vc = to_virt_chan(chan);

	spin_lock_irqsave(&vc->lock, flags);

	mc = to_milbeaut_hdmac_chan(vc);
	md = mc->md;
	if ((md) && (md->dir == DMA_MEM_TO_MEM)) {
		spin_unlock_irqrestore(&vc->lock, flags);
		return stat;
	}

	/* residue from the on-flight chunk */
	if (mc->md && mc->md->vd.tx.cookie == cookie) {
		struct scatterlist *sg;
		u32 done;


		sg = &md->sgl[md->sg_cur];

		if (md->dir == DMA_DEV_TO_MEM)
			done = dch_readl_relaxed(mc->reg_ch_base
					+ MLB_HDMAC_DMACDA,
					MLB_HDMAC_DMACDA,
					mc->mdev);
		else
			done = dch_readl_relaxed(mc->reg_ch_base
					+ MLB_HDMAC_DMACSA,
					MLB_HDMAC_DMACSA,
					mc->mdev);
		done -= sg_dma_address(sg);

		txstate->residue = -done;
	}

	if (!md) {
		vd = vchan_find_desc(vc, cookie);
		if (vd)
			md = to_milbeaut_hdmac_desc(vd);
	}

	if (md) {
		/* residue from the queued chunks */
		for (i = md->sg_cur; i < md->sg_len; i++)
			txstate->residue += sg_dma_len(&md->sgl[i]);
	}

	spin_unlock_irqrestore(&vc->lock, flags);

	return stat;
}

static void milbeaut_hdmac_issue_pending(struct dma_chan *chan) {
	struct virt_dma_chan *vc = to_virt_chan(chan);
	struct milbeaut_hdmac_chan *mc = to_milbeaut_hdmac_chan(vc);
	unsigned long flags;

	spin_lock_irqsave(&vc->lock, flags);

	if (vchan_issue_pending(vc) && !mc->md)
		milbeaut_hdmac_start(mc);

	spin_unlock_irqrestore(&vc->lock, flags); }

static void milbeaut_hdmac_desc_free(struct virt_dma_desc *vd) {
	struct milbeaut_hdmac_desc *md = to_milbeaut_hdmac_desc(vd);

	kfree(md->sgl);
	kfree(md->sgl_dst);
	kfree(md);
}

static struct dma_chan *
milbeaut_hdmac_xlate(struct of_phandle_args *dma_spec, struct of_dma
*of_dma) {
	struct milbeaut_hdmac_device *mdev = of_dma->of_dma_data;
	struct milbeaut_hdmac_chan *mc;
	struct virt_dma_chan *vc;
	struct dma_chan *chan;

	if (dma_spec->args_count != 1)
		return NULL;

	chan = dma_get_any_slave_channel(&mdev->ddev);
	if (!chan)
		return NULL;

	vc = to_virt_chan(chan);
	mc = to_milbeaut_hdmac_chan(vc);
	mc->slave_id = dma_spec->args[0];

	return chan;
}

static int milbeaut_hdmac_chan_init(struct platform_device *pdev,
				    struct milbeaut_hdmac_device *mdev,
				    int chan_id)
{
	struct device *dev = &pdev->dev;
	struct milbeaut_hdmac_chan *mc = &mdev->channels[chan_id];
	char *irq_name;
	int irq, ret;

	irq = platform_get_irq(pdev, chan_id);
	if (irq < 0) {
		dev_err(&pdev->dev, "failed to get IRQ number for ch%d\n",
			chan_id);
		return irq;
	}

	irq_name = devm_kasprintf(dev, GFP_KERNEL, "milbeaut-hdmac-%d",
				  chan_id);
	if (!irq_name)
		return -ENOMEM;

	ret = devm_request_irq(dev, irq, milbeaut_hdmac_interrupt,
			       IRQF_SHARED, irq_name, mc);
	if (ret)
		return ret;

	mc->mdev = mdev;
	mc->reg_ch_base = mdev->reg_base + MLB_HDMAC_CH_STRIDE * (chan_id + 1);
	/* Absorb the difference between M10V and M20V */
	if(mdev->soc_data->number == MLB_M20V_CODE)
		mc->reg_ch_base = mdev->reg_base + MLB_M20V_HDMAC_CH_BASE
		+ MLB_M20V_HDMAC_CH_STRIDE * chan_id;
	mc->vc.desc_free = milbeaut_hdmac_desc_free;
	vchan_init(&mc->vc, &mdev->ddev);

	return 0;
}

static int milbeaut_hdmac_probe(struct platform_device *pdev) {
	struct device *dev = &pdev->dev;
	struct milbeaut_hdmac_device *mdev;
	struct dma_device *ddev;
	struct resource *res;
	const struct product_code *soc_data;
	int nr_chans, ret, i;

	soc_data = device_get_match_data(dev);
	if (!soc_data)
		return -EINVAL;

	nr_chans = platform_irq_count(pdev);
	if (nr_chans < 0)
		return nr_chans;

	ret = dma_set_mask(dev, DMA_BIT_MASK(32));
	if (ret)
		return ret;

	mdev = devm_kzalloc(dev, struct_size(mdev, channels, nr_chans),
			    GFP_KERNEL);
	if (!mdev)
		return -ENOMEM;

	mdev->soc_data = soc_data;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	mdev->reg_base = devm_ioremap_resource(&pdev->dev, res);
	if (IS_ERR(mdev->reg_base))
		return PTR_ERR(mdev->reg_base);

	mdev->clk = devm_clk_get(dev, NULL);
	if (IS_ERR(mdev->clk)) {
		dev_err(dev, "failed to get clock\n");
		return PTR_ERR(mdev->clk);
	}

	ret = clk_prepare_enable(mdev->clk);
	if (ret)
		return ret;

	ddev = &mdev->ddev;
	ddev->dev = dev;
	dma_cap_set(DMA_MEMCPY_SG, ddev->cap_mask);
	dma_cap_set(DMA_SLAVE, ddev->cap_mask);
	dma_cap_set(DMA_PRIVATE, ddev->cap_mask);
	ddev->src_addr_widths = MLB_HDMAC_BUSWIDTHS;
	ddev->dst_addr_widths = MLB_HDMAC_BUSWIDTHS;
	ddev->directions = BIT(DMA_MEM_TO_DEV) | BIT(DMA_DEV_TO_MEM) | BIT(DMA_MEM_TO_MEM);
	ddev->device_free_chan_resources = milbeaut_hdmac_free_chan_resources;
	ddev->device_config = milbeaut_hdmac_chan_config;
	ddev->device_pause = milbeaut_hdmac_chan_pause;
	ddev->device_resume = milbeaut_hdmac_chan_resume;
	ddev->device_prep_slave_sg = milbeaut_hdmac_prep_slave_sg;
	ddev->device_prep_dma_memcpy_sg = milbeaut_hdmac_prep_memcpy_sg;
	ddev->device_terminate_all = milbeaut_hdmac_terminate_all;
	ddev->device_synchronize = milbeaut_hdmac_synchronize;
	ddev->device_tx_status = milbeaut_hdmac_tx_status;
	ddev->device_issue_pending = milbeaut_hdmac_issue_pending;
	INIT_LIST_HEAD(&ddev->channels);

	for (i = 0; i < nr_chans; i++) {
		ret = milbeaut_hdmac_chan_init(pdev, mdev, i);
		if (ret)
			goto disable_clk;
	}

	ret = dma_async_device_register(ddev);
	if (ret)
		goto disable_clk;

	ret = of_dma_controller_register(dev->of_node,
					 milbeaut_hdmac_xlate, mdev);
	if (ret)
		goto unregister_dmac;

	platform_set_drvdata(pdev, mdev);

	return 0;

unregister_dmac:
	dma_async_device_unregister(ddev);
disable_clk:
	clk_disable_unprepare(mdev->clk);

	return ret;
}

static int milbeaut_hdmac_remove(struct platform_device *pdev) {
	struct milbeaut_hdmac_device *mdev = platform_get_drvdata(pdev);
	struct dma_chan *chan;
	int ret;

	/*
	 * Before reaching here, almost all descriptors have been freed by the
	 * ->device_free_chan_resources() hook. However, each channel might
	 * be still holding one descriptor that was on-flight at that moment.
	 * Terminate it to make sure this hardware is no longer running. Then,
	 * free the channel resources once again to avoid memory leak.
	 */
	list_for_each_entry(chan, &mdev->ddev.channels, device_node) {
		ret = dmaengine_terminate_sync(chan);
		if (ret)
			return ret;
		milbeaut_hdmac_free_chan_resources(chan);
	}

	of_dma_controller_free(pdev->dev.of_node);
	dma_async_device_unregister(&mdev->ddev);
	clk_disable_unprepare(mdev->clk);

	return 0;
}

static int milbeaut_hdmac_suspend(struct device *dev)
{
	struct milbeaut_hdmac_device *mdev = dev_get_drvdata(dev);

	clk_disable_unprepare(mdev->clk);

	return 0;
}

static int milbeaut_hdmac_resume(struct device *dev)
{
	struct milbeaut_hdmac_device *mdev = dev_get_drvdata(dev);
	int ret;

	ret = clk_prepare_enable(mdev->clk);

	return ret;
}

static const struct dev_pm_ops milbeaut_hdmac_pm = {
	SET_SYSTEM_SLEEP_PM_OPS(milbeaut_hdmac_suspend, milbeaut_hdmac_resume)
};

static const struct of_device_id milbeaut_hdmac_match[] = {
	{	.compatible = "socionext,milbeaut-m10v-hdmac",
		.data = &product_code_m10v},
	{	.compatible = "socionext,milbeaut-m20v-hdmac",
		.data = &product_code_m20v},
	{	.compatible = "socionext,milbeaut-karine-hdmac",
		.data = &product_code_m20v /* Use M20v because M20V and karine are equivalent */},
	{ /* sentinel */ }
};
MODULE_DEVICE_TABLE(of, milbeaut_hdmac_match);

static struct platform_driver milbeaut_hdmac_driver = {
	.probe = milbeaut_hdmac_probe,
	.remove = milbeaut_hdmac_remove,
	.driver = {
		.name = "milbeaut-hdmac",
		.of_match_table = milbeaut_hdmac_match,
		.pm = &milbeaut_hdmac_pm,
	},
};
module_platform_driver(milbeaut_hdmac_driver);

MODULE_DESCRIPTION("Milbeaut HDMAC DmaEngine driver");
MODULE_LICENSE("GPL v2");
