/*
 * Copyright (c) 2013-     Panasonic Corporation
 * $Id$
 */
/**
   @file	uhadev.c
   @author  
   @brief   Usb Host Agent driver
*/

#include <linux/printk.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/usb.h>
#include <linux/slab.h>

#include <linux/usb/hcd.h>						/* for test */

#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/uaccess.h>

#include <linux/uha/uhadev.h>

#define	CONFIG_USB_PANASONIC_LIMITDEV_NUM	1

/*---------------------------------------------------------------------------*/
/**
 * 
 */
#define UHADEV_MODNAME			"uhadev"
#define UHADEV_VERSION			"0.20"
#define	UHADEV_MAJOR			121

static int uhadev_major = UHADEV_MAJOR;
module_param(uhadev_major, int, S_IRUGO); /* @see: /sys/module/uhadev/parameters/uhadev_major */

#define UHADEV_MINOR_NUM		CONFIG_USB_PANASONIC_LIMITDEV_NUM
#if UHADEV_MINOR_NUM == 1
#define UHADEV_NOTIFY_HUB					  /* HUB: Notify attach, detach */
#endif	/* UHADEV_MINOR_NUM == 1 */

/*---------------------------------------------------------------------------*/
/**
 * 
 */
#define UHADEV_MAX_BUFLEN		sizeof(USBHD_EVENT)

#define UHADEV_RDC_CLEAR			(0x0000)	/* all clear */
#define UHADEV_RDC_RESPONSE			(0x0001)	/* request: response */
#define UHADEV_RDC_NOTIFY_ATTACH	(0x0010)	/* notify: attach */
#define UHADEV_RDC_NOTIFY_DETACH	(0x0020)	/* notify: detach */
#define UHADEV_RDC_NOTIFY_START		(0x0100)	/* notify: start */
#define UHADEV_RDC_NOTIFY_STOP		(0x0200)	/* notify: stop */
#define UHADEV_RDC_NOTIFY_EMERGENCY	(0x8000)	/* notify: emergency */
#define UHADEV_RDC_NOTIFY_MASK		(UHADEV_RDC_NOTIFY_ATTACH |		\
									 UHADEV_RDC_NOTIFY_DETACH |		\
									 UHADEV_RDC_NOTIFY_START |		\
									 UHADEV_RDC_NOTIFY_STOP	 |		\
									 UHADEV_RDC_NOTIFY_EMERGENCY)

USBHD_EVENT uhadev_buf_request[UHADEV_MINOR_NUM];  /* request */
USBHD_EVENT uhadev_buf_response[UHADEV_MINOR_NUM]; /* response */
USBHD_EVENT uhadev_buf_notify[UHADEV_MINOR_NUM];   /* notify */

struct usb_device *uhadev_usbdev_list[UHADEV_MINOR_NUM]; /* usb device list */


struct uhadev_device
{
	USBHD_EVENT *request;						/* request */
	USBHD_EVENT *response;						/* response */
	USBHD_EVENT *notify;						/* notify */
	
	int read_condition;							/* condition for read */
	wait_queue_head_t read_queue;				/* wait queue for read */
	
	struct mutex read_mutex;					/* mutex for read */
	struct mutex write_mutex;					/* mutex for write */
	
	struct usb_device **usbdev_list;			/* usb device struct */
	struct usb_device *udev;					/* usb device struct, for test */
	
	struct cdev chrdev;							/* character device struct */
};

struct uhadev_device uhadev_device[UHADEV_MINOR_NUM];



/*---------------------------------------------------------------------------*/
/**
 * debug
 */
//#define UHADEV_DEBUG

#ifdef UHADEV_DEBUG
	#define UHADEV_DEBUG_TRACE
	#define UHADEV_DEBUG_WARNING
	#define UHADEV_DEBUG_DEBUG
//	#define UHADEV_DEBUG_DUMP_ICC
#endif	/* UHADEV_DEBUG */

#ifdef UHADEV_DEBUG_TRACE
#define uhadev_trace(fmt, args...)		printk(KERN_INFO    "uhadev T:" fmt , ## args)
#else
#define uhadev_trace(fmt, args...)		do {} while (0)
#endif

#ifdef UHADEV_DEBUG_WARNING
#define uhadev_warning(fmt, args...)	printk(KERN_WARNING "uhadev W:" fmt , ## args)
#else
#define uhadev_warning(fmt, args...)	do {} while (0)
#endif

#ifdef UHADEV_DEBUG_DEBUG
#define uhadev_debug(fmt, args...)		printk(KERN_DEBUG   "uhadev D:" fmt , ## args)
#else
#define uhadev_debug(fmt, args...)		do {} while (0)
#endif

#ifdef UHADEV_DEBUG_DUMP_ICC
#define uhadev_dump_icc(buff,size)		{								\
		if (buff) {														\
			int ii;														\
			for (ii = 0; ii < size; ii++) {								\
				printk( KERN_INFO "\t[%3d]: %02X (%3u)\n", ii,			\
						*((unsigned char *)buff+ii),					\
						*((unsigned char *)buff+ii) );					\
			}															\
		}																\
	}
#else
#define uhadev_dump_icc(buff,size)		do {} while (0)
#endif

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static inline int uhadev_coredev_list_search( struct usb_device **usbdev_list, int devnum )
{
	struct usb_device *usbdev = *usbdev_list;
	int minor;
	
	for (minor = 0; minor < UHADEV_MINOR_NUM; minor++)
	{
		if (usbdev == NULL)
		{
			if (devnum < 0)
			{
				return minor;
			}
			break;
		}
		if (usbdev->devnum == devnum)
		{
			return minor;
		}
		usbdev++;
	}
	
	return -ENODEV;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static inline int uhadev_coredev_list_add( struct usb_device **usbdev_list, struct usb_device *usbdev )
{
	int minor;
	
	minor = uhadev_coredev_list_search( usbdev_list, -1 );
	if(minor < 0)
	{
		return -ENOMEM;
	}
	
	usbdev_list[minor] = usbdev;
	
	return minor;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static inline int uhadev_coredev_list_delete( struct usb_device **usbdev_list, struct usb_device *usbdev )
{
	int minor;
	
	minor = uhadev_coredev_list_search( usbdev_list, usbdev->devnum );
	if(minor < 0)
	{
		return -ENODEV;
	}
	
	usbdev_list[minor] = NULL;
	
	return minor;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note usb_control_msg
*/
int uhadev_coredev_submit_ctrl( const USBHD_REQUEST_CTRL *request, USBHD_RESPONSE_CTRL *response, struct usb_device *usbdev )
{
	int ret = 0;
	struct usb_ctrlrequest ctrlmsg;
	unsigned int length;
	unsigned char *pbuff = NULL;
	int pipe;
	int timeout = request->header.submit_timeout;
	
	uhadev_trace( "%s( request= 0x%llx, response= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)request, (u64)response, (u64)usbdev );
	
	if (usbdev == NULL)
	{
		ret = -ENODEV;
		goto finish;
	}
	
	memcpy( &ctrlmsg, request->payload, sizeof(struct usb_ctrlrequest) );
	
	length = ctrlmsg.wLength;	/* To suppress 64k PAGE_SIZE warning */
//	if (length > PAGE_SIZE)
	if (length > sizeof(response->payload))		/* @todo: ??? */
	{
		ret = -EINVAL;
		goto finish;
	}
	pbuff = (unsigned char *) __get_free_page( GFP_KERNEL );
	if (pbuff == NULL)
	{
		ret = -ENOMEM;
		goto finish;
	}
	
	if (ctrlmsg.bRequestType & USB_DIR_IN)		/* IN */
	{
		pipe = usb_rcvctrlpipe( usbdev, request->header.endpoint_number );
	}
	else										/* OUT */
	{
		if (length)
		{
			memcpy( pbuff, &request->payload[sizeof(struct usb_ctrlrequest)], length);
		}
		pipe = usb_sndctrlpipe( usbdev, request->header.endpoint_number );
	}
	
	ret = usb_control_msg( usbdev, pipe, 
						   ctrlmsg.bRequest, ctrlmsg.bRequestType, ctrlmsg.wValue, 
						   ctrlmsg.wIndex, pbuff, ctrlmsg.wLength, timeout );
	if (ret < 0)
	{
		uhadev_warning( "%d = usb_control_msg() failed\n", ret );
		goto finish;
	}
	
	response->header.length = (unsigned char) sizeof(USBHD_RESPONSE_HEADER);
	if (ctrlmsg.bRequestType & USB_DIR_IN)		/* IN */
	{
		if ((ret > 0) && length)
		{
			response->header.length += ret;
			memcpy( response->payload, pbuff, ret );
			response->header.buff_addr = 0;
			response->header.buff_size = ret;
		}
		else
		{
			response->header.buff_addr = 0;
			response->header.buff_size = 0;
		}
	}
	else										/* OUT */
	{
		response->header.buff_addr = 0;
		response->header.buff_size = 0;
	}
	
	ret = 0;
	
finish:
	if (pbuff != NULL)
	{
		free_page( (unsigned long) pbuff );
	}
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static int _uhadev_coredev_submit_bulk( const USBHD_REQUEST_BULK *request, USBHD_RESPONSE_BULK *response, struct usb_device *usbdev )
{
	int ret = 0;
	unsigned char *pbuff_base = NULL;
	unsigned char *pbuff      = NULL;
	unsigned char *kbuff      = NULL;
	unsigned long kbuff_size  = 0;
	unsigned long buff_addr   = 0;
	unsigned long buff_size   = 0;
	int actual_buff_size_all  = 0;
	int actual_buff_size      = 0;
	int pipe = 0;
	unsigned int maxpacket = 0;
	int timeout = request->header.submit_timeout;
	
	uhadev_trace( "%s( request= 0x%llx, response= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)request, (u64)response, (u64)usbdev );
	
	if (usbdev == NULL)
	{
		ret = -ENODEV;
		goto finish;
	}
	
	if (request->header.subcommand == USBHD_DIR_DEV2HOST) /* IN */
	{
		pipe = usb_rcvbulkpipe( usbdev, request->header.endpoint_number );
		maxpacket = usb_maxpacket( usbdev, pipe, 0 );
	}
	else										/* OUT */
	{
		pipe = usb_sndbulkpipe( usbdev, request->header.endpoint_number );
		maxpacket = usb_maxpacket( usbdev, pipe, 1 );
	}
	
	buff_size = request->header.buff_size;
	if (!buff_size)
	{
		ret = -EINVAL;
		uhadev_warning( "\tbuff_size= 0x%lx ) failed\n", buff_size );
		goto finish;
	}
	
	buff_addr = request->header.buff_addr;
//	buff_addr &= ~0xC0000000;					/* @attention: for V8FHD */
	
remain:
	if (buff_addr)
	{
		unsigned long offset = buff_addr & ~PAGE_MASK;
		pbuff_base = (unsigned char *) ioremap_nocache( buff_addr - offset, buff_size + offset );
		if (!pbuff_base)
		{
			uhadev_warning( "\tioremap_nocache( 0x%lx, 0x%lx ) failed\n", buff_addr - offset,  buff_size + offset );
			ret = -EINVAL;
			goto finish;
		}
		pbuff = pbuff_base + offset;
	}
	uhadev_debug( "\tpbuff= 0x%llx <-- addr= 0x%lx, size= 0x%lx )\n", (u64)pbuff, buff_addr, buff_size );
	
	kbuff_size = buff_size;
	kbuff = (unsigned char *) kmalloc( kbuff_size, GFP_KERNEL );
	if (!kbuff)
	{
		uhadev_warning( "\tkmalloc( 0x%lx, GFP_KERNEL ) failed\n", kbuff_size );
		ret = -ENOMEM;
		goto finish;
	}
	uhadev_debug( "\tkbuff= 0x%llx <-- addr= 0x%lx, size= 0x%lx )\n", (u64)kbuff, buff_addr, kbuff_size );
	
	if (usb_pipeout( pipe ))
	{
		if (!buff_addr)
			pbuff = (unsigned char *) &request->payload[actual_buff_size_all];
		memcpy( kbuff, pbuff, kbuff_size );
		uhadev_debug( "\tmemcpy( k:0x%llx, p:0x%llx, 0x%lx )\n", (u64)kbuff, (u64)pbuff, kbuff_size );
	}
	
	actual_buff_size = 0;	
	ret = usb_bulk_msg( usbdev, pipe, kbuff, kbuff_size, &actual_buff_size, timeout );
	uhadev_debug( "%d = usb_bulk_msg(, pipe= 0x%x, buff= 0x%llx, size= 0x%lx, *actual= 0x%x, timeout= %d )\n", 
				  ret, pipe, (u64)kbuff, kbuff_size, actual_buff_size, timeout );
	if (ret < 0)
	{
		uhadev_warning( "%d = usb_bulk_msg(, pipe= 0x%x, buff= 0x%llx, size= 0x%lx, *actual= 0x%x, timeout= %d ) failed\n", 
						ret, pipe, (u64)kbuff, kbuff_size, actual_buff_size, timeout );
		goto finish;
	}
	
	if (ret == 0)
	{
		if (usb_pipein( pipe ))
		{
			if (actual_buff_size > 0)
			{
				if (!buff_addr)
					pbuff = (unsigned char *) &response->payload[actual_buff_size_all];
				memcpy( pbuff, kbuff, actual_buff_size );
				uhadev_debug( "\tmemcpy( p:0x%llx, k:0x%llx, 0x%x )\n", (u64)pbuff, (u64)kbuff, actual_buff_size );
			}
			else if (actual_buff_size == 0)
			{
				/* @example: memory card reader/writer, READ_CAPACITY(10) at none memory card. */
				uhadev_warning( "supported none stall, after IN null packet.\n"); /*  */
				actual_buff_size = kbuff_size;	/* later CSW failed. */
			}
		}
		
		actual_buff_size_all += actual_buff_size;
	}
	
finish:
	if (kbuff)
	{
		kfree( kbuff );
	}
	if (pbuff_base)
	{
		iounmap( pbuff_base );
	}
	if ((ret == 0) && (buff_size > actual_buff_size))
	{
		buff_size -= actual_buff_size;
		buff_addr += actual_buff_size;
		goto remain;
	}
	
	response->header.buff_size = actual_buff_size_all;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
#define SUBMIT_BULK_SPLIT_SIZE		(0x00020000)
int uhadev_coredev_submit_bulk( const USBHD_REQUEST_BULK *request, USBHD_RESPONSE_BULK *response, struct usb_device *usbdev )
{
	int ret = 0;
	USBHD_REQUEST_BULK req_split = *request;
	unsigned long buff_addr = request->header.buff_addr;
	unsigned long buff_size = request->header.buff_size;
	unsigned long actual_buff_size = 0;
	unsigned long actual_buff_size_all = 0;
	
	uhadev_trace( "%s( request= 0x%llx, response= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)request, (u64)response, (u64)usbdev );
	
	if (usbdev == NULL)
	{
		ret = -ENODEV;
		goto finish;
	}
	
	do {
		buff_addr = buff_addr + actual_buff_size;
		req_split.header.buff_addr = buff_addr;
		
		buff_size -= actual_buff_size;
		if (buff_size < SUBMIT_BULK_SPLIT_SIZE)
			req_split.header.buff_size = buff_size;
		else
			req_split.header.buff_size = SUBMIT_BULK_SPLIT_SIZE;
		
		ret = _uhadev_coredev_submit_bulk( &req_split, response, usbdev );
		if (ret < 0)
		{
			goto finish;
		}
		
		actual_buff_size = response->header.buff_size;
		actual_buff_size_all += actual_buff_size;
		
	} while (buff_size > actual_buff_size);
	
finish:
	response->header.buff_size = actual_buff_size_all;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
int uhadev_coredev_submit_clearhalt( const USBHD_REQUEST_CLEARHALT *request, USBHD_RESPONSE_CLEARHALT *response, struct usb_device *usbdev )
{
	int ret = 0;
	int pipe;
	
	uhadev_trace( "%s( request= 0x%llx, response= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)request, (u64)response, (u64)usbdev );
	
	if (usbdev == NULL)
	{
		ret = -ENODEV;
		goto finish;
	}
	
	if (request->header.subcommand == USBHD_DIR_DEV2HOST) /* IN */
	{
		pipe = usb_rcvbulkpipe( usbdev, request->header.endpoint_number );
	}
	else										/* OUT */
	{
		pipe = usb_sndbulkpipe( usbdev, request->header.endpoint_number );
	}
	
	ret = usb_clear_halt( usbdev, pipe );
	if (ret < 0)
	{
		uhadev_warning( "%d = usb_clear_halt() failed\n", ret );
	}
	
finish:
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
int uhadev_coredev_submit_test( const USBHD_REQUEST_TEST *request, USBHD_RESPONSE_TEST *response, struct usb_device *usbdev, u16 typeReq )
{
	int ret = 0;
	struct usb_hcd *hcd;
	
	uhadev_trace( "%s( request= 0x%llx, response= 0x%llx, usbdev= 0x%llx ) ... testmode= %02x typeReq= %04x\n", __func__, (u64)request, (u64)response, (u64)usbdev, request->header.subcommand, typeReq );
	
	if (usbdev == NULL)
	{
		ret = -ENODEV;
		goto finish;
	}
	
	hcd = bus_to_hcd(usbdev->bus);
	if (hcd && hcd->driver && hcd->driver->hub_control)
	{
		ret = hcd->driver->hub_control( hcd, typeReq, USB_PORT_FEAT_TEST,
										request->header.subcommand << 8 | (usbdev->portnum & 0xff), NULL, 0 );
		if (ret < 0)
		{
			uhadev_warning( "%d = hcd->driver->hub_control() failed\n", ret );
		}
	}
	else
	{
		uhadev_warning( "hcd                      = 0x%llx\n", (u64)hcd );
		uhadev_warning( "hcd->driver              = 0x%llx\n", (u64)hcd->driver );
		uhadev_warning( "hcd->driver->hub_control = 0x%llx\n", (u64)hcd->driver->hub_control );
	}
	
finish:
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
int uhadev_coredev_submit( struct uhadev_device *device, struct usb_device **usbdev_list )
{
	int ret = 0;
	const USBHD_EVENT *event = device->request;
	USBHD_EVENT *revent = device->response;
	struct usb_device *usbdev = NULL;
	
	uhadev_trace( "%s( event= 0x%llx, revent= 0x%llx, usbdev_list= 0x%llx )\n", __func__, (u64)event, (u64)revent, (u64)usbdev_list );
	
	memcpy( revent, event, sizeof(USBHD_EVENT) );
	
	/* request */
	{
		switch (event->common.header.command)
		{
		case USBHD_CMD_START:
		case USBHD_CMD_STOP:
			ret = 1;							/* --> notify */
			break;
		case USBHD_CMD_CTRL:
			ret = uhadev_coredev_list_search( usbdev_list, event->request.ctrl.header.address );
			if(ret < 0)
			{
				uhadev_warning( "uhadev_coredev_list_search() failed\n" );
				break;
			}
			usbdev = *(usbdev_list + ret);
			ret = uhadev_coredev_submit_ctrl( &event->request.ctrl, &revent->response.ctrl, usbdev );
			if (ret < 0)
			{
				uhadev_warning( "uhadev_coredev_submit_ctrl() failed (%d)\n", ret );
			}
			break;
		case USBHD_CMD_BULK:
			ret = uhadev_coredev_list_search( usbdev_list, event->request.bulk.header.address );
			if(ret < 0)
			{
				uhadev_warning( "uhadev_coredev_list_search() failed\n" );
				break;
			}
			usbdev = *(usbdev_list + ret);
			ret = uhadev_coredev_submit_bulk( &event->request.bulk, &revent->response.bulk, usbdev );
			if (ret < 0)
			{
				uhadev_warning( "uhadev_coredev_submit_bulk() failed (%d)\n", ret );
			}
			break;
		case USBHD_CMD_CLEARHALT:
			ret = uhadev_coredev_list_search( usbdev_list, event->request.clearhalt.header.address );
			if(ret < 0)
			{
				uhadev_warning( "uhadev_coredev_list_search() failed\n" );
				break;
			}
			usbdev = *(usbdev_list + ret);
			ret = uhadev_coredev_submit_clearhalt( &event->request.clearhalt, &revent->response.clearhalt, usbdev );
			if (ret < 0)
			{
				uhadev_warning( "uhadev_coredev_submit_clearhalt() failed (%d)\n", ret );
			}
			break;
		case USBHD_CMD_ENTER_TEST_MODE:
			usbdev = device->udev;
			ret = uhadev_coredev_submit_test( &event->request.test, &revent->response.test, usbdev, SetPortFeature );
			if (ret < 0)
			{
				uhadev_warning( "uhadev_coredev_submit_test() SetPortFeature failed (%d)\n", ret );
			}
			break;
		case USBHD_CMD_EXIT_TEST_MODE:
			usbdev = device->udev;
			ret = uhadev_coredev_submit_test( &event->request.test, &revent->response.test, usbdev, ClearPortFeature );
			if (ret < 0)
			{
				uhadev_warning( "uhadev_coredev_submit_test() ClearPortFeature failed (%d)\n", ret );
			}
			break;
		default:
			ret = -EINVAL;
			break;
		}
	}
	
	/* response */
	if (ret < 0)
	{
		switch (ret)
		{
		case -EPIPE:
			revent->common.header.status = USBHD_STS_STALLED;
			uhadev_warning( "USBHD_STS_STALLED\n" );
			break;
		case -ETIMEDOUT:
			revent->common.header.status = USBHD_STS_TIMEOUT;
			uhadev_warning( "USBHD_STS_TIMEOUT\n" );
			break;
		case -EINVAL:
			revent->common.header.status = USBHD_STS_INVAL;
			uhadev_warning( "USBHD_STS_INVAL\n" );
			break;
		case -ENOMEM:
		case -EFAULT:
		default:
			if (ret < 0)
			{
				revent->common.header.status = USBHD_STS_OTHER;
				uhadev_warning( "USBHD_STS_OTHER\n" );
			}
			break;
		}
	}
	else
	{
		revent->common.header.status = USBHD_STS_COMPLETION;
	}
	
	if (ret > 0)								/* --> notify */
	{
		;
	}
	else
	{
		device->read_condition |= UHADEV_RDC_RESPONSE;
		wake_up_interruptible( &device->read_queue );
	}
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static 
int uhadev_open( struct inode *inode, struct file *filp ) 
{
	int ret = 0;
	struct uhadev_device *device = container_of( inode->i_cdev, struct uhadev_device, chrdev );
	int minor;
	
	if (device)
	{
		minor = MINOR( device->chrdev.dev );
		uhadev_debug( "%s:%d: %s()\n", UHADEV_MODNAME, minor, __func__ );
		
		filp->private_data = device;
	}
	else
	{
		ret = -ENODEV;
	}
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static 
int uhadev_release( struct inode* inode, struct file* filp )
{
	int ret = 0;
	struct uhadev_device *device = filp->private_data;
	int minor;
	
	if (device)
	{
		minor = MINOR( device->chrdev.dev );
		uhadev_debug( "%s:%d: %s()\n", UHADEV_MODNAME, minor, __func__ );
		
		filp->private_data = 0;
	}
	else
	{
		;										/* not open */
	}
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static 
ssize_t uhadev_read( struct file* filp, char* buf, size_t count, loff_t* offset )
{
	struct uhadev_device *device = filp->private_data;
	int minor;
	ssize_t len = (ssize_t) count;
	int err = 0;
	
	if (!device)
	{
		err = -EFAULT;
		goto failure_err;
	}
	minor = MINOR( device->chrdev.dev );
	uhadev_trace( "%s:%d: %s( count= %ld ) ... device->read_condition= 0x%08x\n", UHADEV_MODNAME, minor, __func__, count, device->read_condition );
	
	if (len <= 0)
	{
		err = -EINVAL;
		goto failure_err;
	}
	else if (len > UHADEV_MAX_BUFLEN)
	{
		len = UHADEV_MAX_BUFLEN;
	}
	
	if (!access_ok( VERIFY_WRITE, buf, len ))
	{
		err = -EFAULT;
		uhadev_warning( "access_ok(VERIFY_WRITE) failed\n" );
		goto failure_err;
	}
	
	if (mutex_trylock( &device->read_mutex ) != 0)
	{
		USBHD_EVENT *uhadev_buf;
		
		err = wait_event_interruptible( device->read_queue, device->read_condition ); /* macro */
		if (err < 0)
		{
			uhadev_warning( "wait_event_interruptible() failed\n" );
			goto failure;
		}
		uhadev_trace( "\twakeup, device->read_condition: 0x%08x\n", device->read_condition );
		
		if (device->read_condition & UHADEV_RDC_RESPONSE)
		{
			uhadev_buf = device->response;
			device->read_condition &= ~UHADEV_RDC_RESPONSE; /* clear */
		}
		else if (device->read_condition & UHADEV_RDC_NOTIFY_DETACH)
		{
			uhadev_buf = device->notify;
			device->read_condition &= ~UHADEV_RDC_NOTIFY_DETACH; /* clear */
		}
		else if (device->read_condition & UHADEV_RDC_NOTIFY_ATTACH)
		{
			uhadev_buf = device->notify;
			device->read_condition &= ~UHADEV_RDC_NOTIFY_ATTACH; /* clear */
		}
		else if (device->read_condition & UHADEV_RDC_NOTIFY_STOP)
		{
			uhadev_buf = device->notify;
			device->read_condition &= ~UHADEV_RDC_NOTIFY_STOP; /* clear */
			msleep( 1 ); /* Wait CLK:Off */
		}
		else if (device->read_condition & UHADEV_RDC_NOTIFY_START)
		{
			uhadev_buf = device->notify;
			device->read_condition &= ~UHADEV_RDC_NOTIFY_START; /* clear */
		}
		else if (device->read_condition & UHADEV_RDC_NOTIFY_EMERGENCY)
		{
			uhadev_buf = device->notify;
			device->read_condition &= ~UHADEV_RDC_NOTIFY_EMERGENCY; /* clear */
		}
		else
		{
			err = -EFAULT;
			uhadev_warning( "\tcheck, device->read_condition: 0x%08x\n", device->read_condition );
			device->read_condition = UHADEV_RDC_CLEAR;
			goto failure;
		}
		
		len = uhadev_buf->common.header.length;
		
		uhadev_debug( "\t%s( count= %ld )\n", __func__, len );
		uhadev_dump_icc( uhadev_buf->buffer, len );
		if (copy_to_user( buf, uhadev_buf->buffer, len ) != 0)
		{
			err = -EFAULT;
			uhadev_warning( "copy_to_user() failed\n" );
			goto failure;
		}
	}
	else
	{
		uhadev_warning( "\tbusy, device->read_condition: 0x%08x\n", device->read_condition );
		err = -EBUSY;
		goto failure_err;
	}
failure:
	mutex_unlock( &device->read_mutex );
	if (err < 0)	goto failure_err;
	
	goto finish;
	
failure_err:
	len = err;
	
finish:
	return len;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static 
ssize_t uhadev_write( struct file* filp, const char* buf, size_t count, loff_t* offset )
{
	struct uhadev_device *device = filp->private_data;
	int minor;
	ssize_t len = (ssize_t) count;
	int err = 0;
	
	if (!device)
	{
		err = -EFAULT;
		goto finish;
	}
	minor = MINOR( device->chrdev.dev );
	uhadev_debug( "%s:%d: %s( count= %ld )\n", UHADEV_MODNAME, minor, __func__, count );
	
	if (len <= 0)
	{
		err = -EINVAL;
		goto finish;
	}
	else if (len > UHADEV_MAX_BUFLEN)
	{
		len = UHADEV_MAX_BUFLEN;
	}
	
	if (!access_ok( VERIFY_READ, buf, len ))
	{
		err = -EFAULT;
		uhadev_warning( "access_ok(VERIFY_READ) failed\n" );
		goto finish;
	}
	
	if (copy_from_user( device->request, buf, len ) != 0)
	{
		err = -EFAULT;
		uhadev_warning( "copy_from_user() failed\n" );
		goto finish;
	}
	
	mutex_lock( &device->write_mutex );
	{
		uhadev_dump_icc( device->request, device->request->common.header.length );
		err = uhadev_coredev_submit( device, device->usbdev_list );
		if (err < 0)
		{
			uhadev_warning( "uhadev_coredev_submit() failed\n" );
			goto failure;
		}
	}
failure:
	mutex_unlock( &device->write_mutex );
	
finish:
	if (err < 0)
	{
		len = err;
	}
	return len;
}


/*---------------------------------------------------------------------------*/
/**
 * 
 */
static struct file_operations uhadev_chrdev_fops = {
	.owner		= THIS_MODULE,
	.open		= uhadev_open,
	.release	= uhadev_release,
	.read		= uhadev_read,
	.write		= uhadev_write,
	.unlocked_ioctl	= NULL,
};


/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note attach
*/
int uhadev_coredev_device_attach( USBHD_EVENT *event, struct usb_device *usbdev )
{
	int ret = 0;
	USBHD_NOTIFY_ATTACH *attach	 = &event->notify.attach;
	size_t len; 
	size_t offset;
	
	uhadev_trace( "%s( event= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)event, (u64)usbdev );
	
	/* payload */
	offset = 0;
	{
		unsigned char *dest = &attach->payload[0];
		void *src;
		size_t remain = sizeof(USBHD_NOTIFY_ATTACH) - sizeof(USBHD_NOTIFY_HEADER);
		int cfgidx;
		
		for (cfgidx = -1; (cfgidx < (int) usbdev->descriptor.bNumConfigurations)
				 && (remain > 0); cfgidx++)
		{
			if (cfgidx < 0)							/* device descriptor */
			{
				src = &usbdev->descriptor;
				len = sizeof(struct usb_device_descriptor);
			}
			else								   /* configuration descriptors */
			{
				src = usbdev->rawdescriptors[cfgidx];
				len = usbdev->config[cfgidx].desc.wTotalLength;
			}
			if (len > remain)
			{
				len = remain;
			}
			memcpy( dest + offset, src, len );
			offset += len;
			remain -= len;
		}
	}
	
	/* header */
	len = sizeof(USBHD_NOTIFY_HEADER) + offset;
	{
		USBHD_NOTIFY_HEADER	*header = &attach->header;
		
		header->length = (unsigned char) len;
		header->command = (unsigned char) USBHD_CMD_ATTACH;
		header->status = (unsigned char) usbdev->state;
		header->speed = (unsigned char) usbdev->speed;
		header->address = (unsigned char) usbdev->devnum;
		header->endpoint_number = (unsigned char) 0; /* don't care */
		header->reserved1 = (unsigned char) 0;		 /* don't care */
		header->reserved2 = (unsigned char) 0;		 /* don't care */
		header->buff_addr = 0;
		header->buff_size = offset;
	}
	
	ret = (int) len;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note detach
*/
int uhadev_coredev_device_detach( USBHD_EVENT *event, struct usb_device *usbdev )
{
	int ret = 0;
	USBHD_NOTIFY_DETACH *detach	 = &event->notify.detach;
	size_t len; 
	size_t offset;
	
	uhadev_trace( "%s( event= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)event, (u64)usbdev );
	
	/* payload */
	offset = 0;
	{
		/* none */
	}
	
	/* header */
	len = sizeof(USBHD_NOTIFY_HEADER) + offset;
	{
		USBHD_NOTIFY_HEADER	*header = &detach->header;
		
		header->length = (unsigned char) len;
		header->command = (unsigned char) USBHD_CMD_DETACH;
		header->status = (unsigned char) usbdev->state;
		header->speed = (unsigned char) usbdev->speed;
		header->address = (unsigned char) usbdev->devnum;
		header->endpoint_number = (unsigned char) 0; /* don't care */
		header->reserved1 = (unsigned char) 0;		 /* don't care */
		header->reserved2 = (unsigned char) 0;		 /* don't care */
		header->buff_addr = 0;						 /* don't care */
		header->buff_size = offset;					 /* don't care */
	}
	
	ret = (int) len;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note detach
*/
int uhadev_coredev_device_start( USBHD_EVENT *event, struct usb_device *usbdev )
{
	int ret = 0;
	USBHD_RESPONSE_START *response = &event->response.start;
	size_t len; 
	size_t offset;
	
	uhadev_trace( "%s( event= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)event, (u64)usbdev );
	
	/* payload */
	offset = 0;
	{
		/* none */
	}
	
	/* header */
	len = sizeof(USBHD_RESPONSE_HEADER) + offset;
	{
		USBHD_RESPONSE_HEADER	*header = &response->header;
		
		header->length = (unsigned char) len;
		header->command = (unsigned char) USBHD_CMD_START;
		header->status = (unsigned char) USBHD_STS_COMPLETION;
		header->speed = (unsigned char) usbdev->speed;
		header->address = (unsigned char) usbdev->devnum;
		header->endpoint_number = (unsigned char) 0; /* don't care */
		header->buff_addr = 0;						 /* don't care */
		header->buff_size = offset;					 /* don't care */
	}
	
	ret = (int) len;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note detach
*/
int uhadev_coredev_device_stop( USBHD_EVENT *event, struct usb_device *usbdev )
{
	int ret = 0;
	USBHD_RESPONSE_STOP *response = &event->response.stop;
	size_t len; 
	size_t offset;
	
	uhadev_trace( "%s( event= 0x%llx, usbdev= 0x%llx )\n", __func__, (u64)event, (u64)usbdev );
	
	/* payload */
	offset = 0;
	{
		/* none */
	}
	
	/* header */
	len = sizeof(USBHD_RESPONSE_HEADER) + offset;
	{
		USBHD_RESPONSE_HEADER	*header = &response->header;
		
		header->length = (unsigned char) len;
		header->command = (unsigned char) USBHD_CMD_STOP;
		header->status = (unsigned char) USBHD_STS_COMPLETION;
		header->speed = (unsigned char) usbdev->speed;
		header->address = (unsigned char) usbdev->devnum;
		header->endpoint_number = (unsigned char) 0; /* don't care */
		header->buff_addr = 0;						 /* don't care */
		header->buff_size = offset;					 /* don't care */
	}
	
	ret = (int) len;
	
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
   @see linux/notifier.h
*/
int uhadev_coredev_notify( struct notifier_block *self, unsigned long action, void *dev )
{
	struct usb_device *usbdev = dev;
	struct uhadev_device *device = NULL;
	int minor;
	
	uhadev_trace( "%s( , action= 0x%08lu, )\n", __func__, action );
	if (dev == NULL)
	{
		return NOTIFY_BAD;
	}
	
#if 1
	uhadev_trace( "\t@devnum       : %d\n", usbdev->devnum );
	uhadev_trace( "\t@state        : %d\n", usbdev->state );
	uhadev_trace( "\t@speed        : %d\n", usbdev->speed );
//	uhadev_trace( "\t@tt           : 0x%llx\n", (u64)usbdev->tt );
	uhadev_trace( "\t@parent       : 0x%llx\n", (u64)usbdev->parent );
	uhadev_trace( "\t@bus          : 0x%llx\n", (u64)usbdev->bus );
//	uhadev_trace( "\t@config       : 0x%llx\n", (u64)usbdev->config );
	uhadev_trace( "\t@bus_mA       : %d\n", usbdev->bus_mA );
	uhadev_trace( "\t@level        : %d\n", usbdev->level );
	uhadev_trace( "\t@can_submit   : %d\n", usbdev->can_submit );
//	uhadev_trace( "\t@product      : %s\n", usbdev->product );
//	uhadev_trace( "\t@manufacturer : %s\n", usbdev->manufacturer );
//	uhadev_trace( "\t@serial       : %s\n", usbdev->serial );
	uhadev_trace( "\t@maxchild     : %d\n", usbdev->maxchild );
//	uhadev_trace( "\t@urbnum       : %d\n", usbdev->urbnum.counter );
	uhadev_trace( "\t@slot_id      : %d\n", usbdev->slot_id );
#endif
	
	switch (action)
	{
	case USB_DEVICE_ADD:
		if (usbdev->level > 0)					/* device */
		{
			if (usbdev->maxchild > 0)			/* hub ? */
			{
#ifndef UHADEV_NOTIFY_HUB
				return NOTIFY_OK;
#endif	/* !UHADEV_NOTIFY_HUB */
			}
			minor = uhadev_coredev_list_add( uhadev_usbdev_list, usbdev );
			if (minor < 0)
			{
				return NOTIFY_BAD;
			}
			device = &uhadev_device[minor];
			device->udev = usbdev;
			uhadev_coredev_device_attach( device->notify, usbdev );
			device->read_condition |= UHADEV_RDC_NOTIFY_ATTACH;
		}
		if (usbdev->level == 0)					/* root hub */
		{
			minor = 0;
			device = &uhadev_device[minor];
			uhadev_coredev_device_start( device->notify, usbdev );
			device->read_condition |= UHADEV_RDC_NOTIFY_START;
		}
		break;
	case USB_DEVICE_REMOVE:
		if (usbdev->level > 0)					/* device */
		{
			if (usbdev->maxchild > 0)			/* hub ? */
			{
#ifndef UHADEV_NOTIFY_HUB
				return NOTIFY_OK;
#endif	/* !UHADEV_NOTIFY_HUB */
			}
			minor = uhadev_coredev_list_delete( uhadev_usbdev_list, usbdev );
			if (minor < 0)
			{
				return NOTIFY_BAD;
			}
			device = &uhadev_device[minor];
			device->udev = NULL;
			uhadev_coredev_device_detach( device->notify, usbdev );
			device->read_condition |= UHADEV_RDC_NOTIFY_DETACH;
		}
		if (usbdev->level == 0)					/* root hub */
		{
			minor = 0;
			device = &uhadev_device[minor];
			uhadev_coredev_device_stop( device->notify, usbdev );
			/* next is USB_BUS_REMOVE */
		}
		break;
	case USB_BUS_ADD:
		return NOTIFY_OK;
	case USB_BUS_REMOVE:
		minor = 0;
		device = &uhadev_device[minor];
		device->read_condition |= UHADEV_RDC_NOTIFY_STOP;
		break;
	default:
		return NOTIFY_OK;
	}
	
	if (device && (device->read_condition & UHADEV_RDC_NOTIFY_MASK))
	{
		wake_up_interruptible( &device->read_queue );
	}
	
	return NOTIFY_OK;
}

static struct notifier_block uhadev_coredev_nb = {
	.notifier_call = uhadev_coredev_notify,
};


/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static int __init uhadev_init( void )
{
	int ret = 0;
	struct uhadev_device *device;
	int cnum;
	
	uhadev_trace( "%s()\n", __func__ );
	
	/* character device */
	{
		dev_t chrdev_id = 0;
		
		if (uhadev_major != 0)
		{
			chrdev_id = MKDEV( uhadev_major, 0 );
			ret = register_chrdev_region( chrdev_id, UHADEV_MINOR_NUM, UHADEV_MODNAME );
			if (ret < 0)
			{
				uhadev_warning( "register_chrdev_region() failed.\n" );
				goto failure;
			}
		}
		else
		{
			ret = alloc_chrdev_region( &chrdev_id, 0, UHADEV_MINOR_NUM, UHADEV_MODNAME );
			if (ret < 0)
			{
				uhadev_warning( "alloc_chrdev_region() failed.\n" );
				goto failure;
			}
			uhadev_major = MAJOR( chrdev_id );
		}
		
		for (cnum = 0; cnum < UHADEV_MINOR_NUM; cnum++)
		{
			device = &uhadev_device[cnum];
			
			device->request = &uhadev_buf_request[cnum];
			device->response = &uhadev_buf_response[cnum];
			device->notify = &uhadev_buf_notify[cnum];
			
			device->read_condition = UHADEV_RDC_CLEAR;
			init_waitqueue_head( &device->read_queue );
			
			mutex_init( &device->read_mutex );
			mutex_init( &device->write_mutex );
			
			uhadev_usbdev_list[cnum] = NULL;
			device->usbdev_list = uhadev_usbdev_list;
			device->udev = NULL;
			
			cdev_init( &device->chrdev, &uhadev_chrdev_fops );
			device->chrdev.owner = THIS_MODULE;
			ret = cdev_add( &device->chrdev, MKDEV( uhadev_major, cnum ), 1 );
			if (ret < 0)
			{
				uhadev_warning( "cdev_add() failed (%d)\n", ret );
				goto failure;
			}
		}
	}
	
	/* usb core */
	{
		usb_register_notify( &uhadev_coredev_nb );
	}
	
	printk( KERN_NOTICE "%s driver. Ver.%s (major:%d)\n", UHADEV_MODNAME, UHADEV_VERSION, uhadev_major );
	goto finish;
	
failure:
	for (cnum = 0; cnum < UHADEV_MINOR_NUM; cnum++)
	{
		device = &uhadev_device[cnum];
		cdev_del( &device->chrdev );
	}
	unregister_chrdev_region( MKDEV( uhadev_major, 0 ), UHADEV_MINOR_NUM );
	
finish:
	return ret;
}

/*---------------------------------------------------------------------------*/
/**
   @brief 
   @param[in] 
   @param[out] 
   @param[in,out] 
   @return 
   @note 
*/
static void __exit uhadev_exit( void )
{
	uhadev_trace( "%s()\n", __func__ );
	
	/* usb core */
	{
		usb_unregister_notify( &uhadev_coredev_nb );
	}
	
	/* character device */
	{
		struct uhadev_device *device;
		int cnum;
		
		for (cnum = 0; cnum < UHADEV_MINOR_NUM; cnum++)
		{
			device = &uhadev_device[cnum];
			cdev_del( &device->chrdev );
		}
		unregister_chrdev_region( MKDEV( uhadev_major, 0 ), UHADEV_MINOR_NUM );
	}
	
	printk( KERN_NOTICE "%s exit.\n", UHADEV_MODNAME );
}

/*---------------------------------------------------------------------------*/
/**
 * 
 */
MODULE_AUTHOR("Panasonic");
MODULE_DESCRIPTION("Usb Host Agent driver");
MODULE_LICENSE("GPL");

module_init( uhadev_init );
module_exit( uhadev_exit );

/*---------------------------------------------------------------------------*/
